/* dz11.c: UNIBUS DZ serial line simulator

   Copyright (c) 1997, 1998
   Thord Nilson, <thordn@stacken.kth.se>

   Commercial use prohibited.

   Hacked to run with Simh2.6 by Art Krewat [AAK] - 05-28-2001

   dz0		8 line terminal interface.


   2001-05-29	[AAK] Renamed to "dz11.c" and hacked for both PDP-10 and PDP-11
   2001-05-28	[AAK] Hacked to run with simh2.6 for the PDP-10
   1999-09-02	Added disconnect log messages.
   1998-09-21	Bugfix: also detect "WILL BINARY"....
   1998-06-12	Detect "WONT BINARY" in telnet negotiation.
   1998-04-16	Added modem control, clock catch-up.
   1997-10-10	Basic version, no modem control etc.

*/

#define DZ_CONNMSG	/* logg connection messages on console.  */
#include <signal.h>
#include <fcntl.h>

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#ifdef sun
# include <sys/file.h>
#endif

#ifdef PDP11
#include "pdp11_defs.h"
#endif
#ifdef PDP10
#include "pdp10_defs.h"
#endif

static int32 modem_control = 0;		/* Flags if modem control is active */
static int32 timer_catchup = 0;		/* if we want the timer to approx right */
static int32 dz0_norm_i_clock = 20000;	/* interval timer for 50 hz line freq */
static int32 dz0_fast_i_clock = 10000;	/* interval timer for time catchup  */
					/* (smaller diff does not work w Linux */
static int32 dz0_slow_i_clock = 30000;	/* interval timer for time slowdown  */
static int32 dz0_current_i_clock = 0;
static char dz0_attach_message[100];

int32 dz0_clock_wait = 5000;		/* Clock wait time. (instructions) */
/*
  			RSTS/E V8.0-06 seems to need atleast 4000 here 
			or it may crash during boot, RSTS/E V9.2 seems
			to make it with as low as 500.
			If you have a SLOW cpu you may want to decrease
			this value so the real-time clock wont lag behind
*/

int32 emulated_time;			/* Holds the emulated time */
extern int32 int_req;

int32 dz0_csr = 0;			/* control/status */
int32 dz0_rbuf = 0;			/* receiver buffer (RO/WordOnly) */
int32 dz0_lpr = 0;			/* line param reg  (WO/WordOnly) */
int32 dz0_tcr = 0;			/* transmit control reg */
int32 dz0_msr = 0;			/* modem status reg */
int32 dz0_tdr = 0;			/* transmit data reg */
int32 printc = 0;
int32 printw = 0;
int32 rline = 0;
int32 i = 0;
/* int32 xdebug = 0;	*/

/* DZ0 queue handling data structures */

#define MAXBUF 128			/* max no of chars to buffer */
int32 connection[9];			/* 8 terminal lines + conn socket */
char  connrxbuff[8][MAXBUF];		/* rx-buffers for dz.		  */
char  conntxbuff[8][MAXBUF];		/* tx-buffers for dz.		  */
int32 connrxbufp1[8];			/* indexes for rx & tx buffers   */
int32 connrxbufp2[8];			/* indexes for rx & tx buffers   */
int32 conntxbufp1[8];			/* indexes for rx & tx buffers   */
int32 conntxbufp2[8];			/* indexes for rx & tx buffers   */
int32 enabled_tx_lines;			/* bitmap for the enabled lines */
int32 telnet_init[8];			/* # of telnet init negotiations */
int32 disabled_telnet_bin;		/* bitmap for telnet lines NOT in */
					/* bin-mode */
struct sockaddr_in connsock[8];		/* save som socket info */

int32 skip_next_char;			/* bitmap for chars to skip. */

int32 DZ0sock;

char mantra[18];


sig_atomic_t dz_io_signal = 0, line_int_signal = 0,
	 external_async_event = 0, xevent = 0, xsecond = 0;



struct sockaddr_in clientname;
struct sigaction new_action, old_action;
struct itimerval old_ti, new_ti;

t_stat dz0_svc (UNIT *uptr);
t_stat dz0_reset (DEVICE *dptr);
t_stat dz0_attach (UNIT *uptr, char *ptr);
t_stat dz0_detach (UNIT *uptr);
/* extern int16 old_clock;		/* nonzero if using old line clock mode */
/* extern int32 pending_clock_int; */

/* DZ0 data structures

   dz0_dev	LPT device descriptor
   dz0_unit	LPT unit descriptor
   dz0_reg	LPT register list
*/

UNIT dz0_unit = {
	UDATA (&dz0_svc, UNIT_ATTABLE, 0), SERIAL_OUT_WAIT };

REG dz0_reg[] = {
	{ ORDATA (CSR,  dz0_csr, 16) },
	{ ORDATA (RBUF, dz0_rbuf, 16) },
	{ ORDATA (LPR,  dz0_lpr, 16) },
	{ ORDATA (TCR,  dz0_tcr, 16) },
	{ ORDATA (MSR,  dz0_msr, 16) },
	{ ORDATA (TDR,  dz0_tdr, 16) },
	{ DRDATA (NICLOCK, dz0_norm_i_clock, 31), PV_LEFT },
	{ DRDATA (FICLOCK, dz0_fast_i_clock, 31), PV_LEFT },
	{ DRDATA (SICLOCK, dz0_slow_i_clock, 31), PV_LEFT },
	{ DRDATA (CLKWAIT, dz0_clock_wait, 31), PV_LEFT },
	{ NULL }  };

DEVICE dz0_dev = {
	"DZ0", &dz0_unit, dz0_reg, NULL,
	1, 10, 31, 1, 8, 8,			/* ??? radix ???? */
	NULL, NULL, &dz0_reset,			/* say we have one unit */
	NULL, &dz0_attach, &dz0_detach };

/****************************************************************/
/* Support routines etc. */
int poll_rx_channels(void);
void poll_tx_channels(void);
void accept_connection(void);

/* Routines for signal handling */

void setup_interval_timer(int interval)
{
  /*  fprintf(stderr, "Setting interval-timer to: %d \r\n", interval); */
  dz0_current_i_clock = interval;
  new_ti.it_interval.tv_usec = interval;
  new_ti.it_interval.tv_sec = 0;
  new_ti.it_value.tv_usec = interval;
  new_ti.it_value.tv_sec = 0;
  
  if (setitimer(ITIMER_REAL, &new_ti, &old_ti) < 0) {
    perror("dz0: setitimer failed.");
    exit(EXIT_FAILURE);
  }
}

void xwrite(int32 xsock, char *message)
{
  write(xsock, message, strlen(message));
}

void update_int_req(void)		/* called from simulator main loop when external_async_event set */
{
  int32 foo;

  external_async_event = 0;
  if (dz_io_signal) {
    dz_io_signal = 0;
    accept_connection();
    if (poll_rx_channels()) {
      dz0_csr |= 0x80;		/* say data available */
      if (dz0_csr & 0x40) {
	int_req |= INT_DZ0RX; 	/* gen int if enabled */
      }
    }
  }
  if (line_int_signal) {
    line_int_signal--;
    poll_tx_channels();
    /* if (old_clock == 0) pending_clock_int++;		/* send clock-tick */
  }
  if (timer_catchup && xsecond) {
    xsecond--;
    emulated_time++;		/* Assume 1 second has passed. */
    foo = time(NULL) - emulated_time; /* get diff */
    /* pending_clock_int += (foo * 50); /* Add clk-ticks for diff */
    emulated_time += foo;	     /* and update emulated time too */
/* #ifdef DZ_DEBUG
    fprintf(stderr, "Time-diff is: %d Pend_int: %d\r\n", foo, pending_clock_int); 
#endif     
    if ((pending_clock_int >= 150) && (dz0_current_i_clock != dz0_fast_i_clock))
      setup_interval_timer(dz0_fast_i_clock);
    
    if ((pending_clock_int <=  50) && (dz0_current_i_clock != dz0_norm_i_clock))
      setup_interval_timer(dz0_norm_i_clock);
    
    if ((pending_clock_int <= -50) && (dz0_current_i_clock != dz0_slow_i_clock))
      setup_interval_timer(dz0_slow_i_clock);
*/
    
  }
}

/* Called when a SIGIO or SIGALRM arrives */

void catch_alarm(int sig)
{
  external_async_event = 1;
  if (sig == SIGIO) dz_io_signal++;
  if (sig == SIGALRM) {
    line_int_signal++;
    if (xevent++ > 50) {
      xevent -= 50;
      xsecond++;
      dz_io_signal++;		/* check i/o every second regardles... */
    }
  }
}

int make_socket(unsigned short int port)
{
  int sock, oldflags;
  struct sockaddr_in name;

  /* create the socket */
  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock < 0 ) {
      perror("dz0: socket");
      return -1;
    }
  /* Give the socket a name */
  name.sin_family = AF_INET;
  name.sin_port = htons(port);
  name.sin_addr.s_addr = htonl(INADDR_ANY);
  if (bind (sock, (struct sockaddr *) &name, sizeof(name)) < 0)
    {
      perror ("dz0: bind");
      return -1;
    }
  oldflags = fcntl(sock, F_GETFL,0);
  fcntl(sock, F_SETFL, oldflags | FASYNC | O_NONBLOCK);
  oldflags = fcntl(sock, F_SETOWN, getpid());

  DZ0sock = sock;

  return sock;
}

void accept_connection(void)
{
  int32 newsock, size, foo;
  
  /* connection request on original socket */
  size = sizeof(clientname);
  newsock = accept (connection[8], (struct sockaddr *) &clientname, &size);
  if (newsock < 0)
    {
      if (errno == EWOULDBLOCK) return;
      perror ("dz0: accept");
      exit (EXIT_FAILURE);
    }
  foo = fcntl(newsock, F_GETFL,0);
  fcntl(newsock, F_SETFL, foo | FASYNC | O_NONBLOCK);
  foo = fcntl(newsock, F_SETOWN, getpid());
  
  foo = 0;
  while ((connection[foo] != 0) && (foo < 8)) foo++;
  if (foo == 8) {
    xwrite(newsock, "All connections busy... please try later\r\n\n");
    close(newsock);
    return;
  }
  connection[foo] = newsock;
  connrxbufp1[foo] = 0;
  connrxbufp2[foo] = 0;
  conntxbufp1[foo] = 0;
  conntxbufp2[foo] = 0;
  telnet_init[foo] = 0;
 
  /*  Here we could manipulate the modem-status signals if we want. */
  if (modem_control) {
    dz0_msr |= (1 << foo);	/* Set Ring-indicator */
  }

  
#ifdef DZ_CONNMSG
  fprintf(stderr,
	  "\r\nServer: connect from host %s, port %hd, sock %d.\r\n", 
	  inet_ntoa(clientname.sin_addr),
	  ntohs (clientname.sin_port), newsock);
  connsock[foo] = clientname;
#endif
  
  send(newsock, mantra, 15, 0);		  
#ifdef PDP11
  xwrite(newsock, "\n\r\nWelcome to the PDP11 simulator.\r\n\n");
#endif
#ifdef PDP10
  xwrite(newsock, "\n\r\nWelcome to the PDP10 simulator.\r\n\n");
#endif
  return;
}
  
int read_from_client(int index)
{
  int nbytes, foo;

  nbytes = read (connection[index], connrxbuff[index], MAXBUF);
  /* printf("read %d bytes errno=%d\n", nbytes,errno); */
  if (nbytes < 0) 
    {
      if (errno = EWOULDBLOCK) return 0;     /* this is when no data avail */
      /* read error */
      perror("dz0: read");
      exit (EXIT_FAILURE);
    }
  else if (nbytes == 0)
    /* End-of-file */
    return -1;
  else
    {
      /* Data read */
      connrxbufp2[index] = nbytes;
      connrxbufp1[index] = 0;
      if (telnet_init[index] < 5) {
	foo = 0;
	while (nbytes-foo  > 2) {
	  if (connrxbuff[index][foo] < 0) {
#ifdef DZ_DEBUG
	    fprintf(stderr, "Telnet_ini: [%d][%d][%d]\r\n", 
		    connrxbuff[index][foo], 
		    connrxbuff[index][foo+1], 
		    connrxbuff[index][foo+2]);
#endif
	    if ((connrxbuff[index][foo+1] == -4) &&	/* Won't [252] */
		(connrxbuff[index][foo+2] == 0)) {	/* bin   */
	      disabled_telnet_bin |= (1 << index);	/* mode  */
	    }
	    if ((connrxbuff[index][foo+1] == -5) &&	/* Will [251] */
		(connrxbuff[index][foo+2] == 0)) {	/* bin   */
	      disabled_telnet_bin &= ~(1 << index);	/* mode  */
	    }
	    connrxbuff[index][foo++] = 0;	/* zap to NULL */
	    connrxbuff[index][foo++] = 0;
	    connrxbuff[index][foo++] = 0;
	    telnet_init[index]++;
	  } else foo++;
	}
      }
      return 1;					/* really got data */
    }
}

int poll_rx_channels(void)			/* returns 1 if got data, 0 otherwise. */
{
  int32 i, gotdata, flag;

  gotdata = 0;
  for (i = 0; i < 8; i++)
    {
      if (connection[i]) {
	if (connrxbufp2[i] == 0) {
	  flag = read_from_client(i);
	  if (flag < 0 ) {	/* if conn closed */
	    close(connection[i]);
	    connection[i] = 0;
	    conntxbufp1[i] = 0;
	    conntxbufp2[i] = 0;
	    if (modem_control) {	/* reset carrier det */
	      dz0_msr &= ~(256 << i);
	    }
	  } else {
	    if (flag == 1) {
	      gotdata = 1;
	    }
	  }
	} else gotdata = 1;		/* else data is already available */
      }
    }
  return gotdata;
}
	  
void poll_tx_channels(void)
{
  int i, nbytes, sentbytes;
  for (i = 0; i < 8; i++)
    {
      if (connection[i]) {
	nbytes = conntxbufp2[i] - conntxbufp1[i];
	if (nbytes) {
/*	  fprintf(stderr, "Sending: %c\r\n", conntxbuff[i][conntxbufp1[i]]); */
	  sentbytes = write(connection[i], &conntxbuff[i][conntxbufp1[i]], nbytes);
	  if (sentbytes == nbytes) {
	    enabled_tx_lines |= 1 << i;		/* enable this line */
	    conntxbufp1[i] = 0;
	    conntxbufp2[i] = 0;
	  } else {
	    conntxbufp1[i] += sentbytes;
	  }
	}
      }
    }
}


void disconn_line(int32 bitmask)
{
  int32 i, ii;

  ii = 1;
  for (i = 0; i < 8; i++) {
    if ((connection[i] != 0) && (bitmask & ii)) {
      xwrite(connection[i], "\r\nLine HANGUP.\r\n");
      close(connection[i]);
      connection[i] = 0;
      conntxbufp1[i] = 0;
      conntxbufp2[i] = 0;
#ifdef DZ_CONNMSG
      fprintf(stderr,
	      "\r\nServer: disconnecting host %s, port %hd.\r\n", 
	      inet_ntoa(connsock[i].sin_addr),
	      ntohs (connsock[i].sin_port));
#endif
    }
    ii = ii << 1;
  }
}



/****************************************************************/
/* Terminal line routines

   dz0_rd	I/O page read
   dz0_wr	I/O page write
   dz0_svc	process event (Not used...)
   dz0_reset	process reset
   dz0_attach	process attach
   dz0_detach	process detach
*/

/* Read from DZ interface */


/* Note that read from odd addresses shall return word data from 
   even address, result is >> 8 by calling routine. Do NOT do this
   here. For write routine this is different. */

t_stat dz0_rd (int32 *data, int32 PA, int32 access)
{
  int i, ii, tmp;

  /* fprintf(stderr, "dz0-rd: PA = %d \r\n", PA & 7);  */

  switch (PA & 7) {			/* decode PA<2:0> */

  case 0:
  case 1:				/* odd address illegal... */
    *data = dz0_csr;
    return SCPE_OK;

  case 2:
  case 3:				/* odd, illegal -- but we do it anyway */
    if ((dz0_csr & 0x80) == 0) { 	/* if read and end_of_data */
      *data = dz0_rbuf = 0;
      return SCPE_OK;
    }
    for (i = 0; i < 8; i++) {
      if (skip_next_char & (1 << i)) {
	ii = connrxbufp2[i] - connrxbufp1[i];
	if (ii > 0) {
	  connrxbufp1[i]++;
	  skip_next_char &= ~(1 << i);
	  if (ii == 1) {		/* reset pointers if buffer empty */
	    connrxbufp1[i] = 0;
	    connrxbufp2[i] = 0;
	  }	
	}
      }

      ii = connrxbufp2[i] - connrxbufp1[i];
      if (ii > 0) {
	tmp = connrxbuff[i][connrxbufp1[i]++];
	*data = 0x8000 | (i << 8) | tmp;

	if ((disabled_telnet_bin & (1 << i)) && tmp == 13) {
	  skip_next_char |= 1 << i;
	}
	
	if (ii == 1) {			/* reset pointers if buffer empty */
	  connrxbufp1[i] = 0;
	  connrxbufp2[i] = 0;
	}
	
	ii = 0;
	for (i = 0; i < 8; i++) ii |= connrxbufp2[i];
	if (ii == 0) {
	  if (poll_rx_channels() == 0) {
	    int_req &= ~INT_DZ0RX;
	    return SCPE_OK;
	  }
	}
	int_req |= INT_DZ0RX;
	return SCPE_OK;
      }
    }
    int_req &= ~INT_DZ0RX;
    return SCPE_OK;

  case 4:
  case 5:
    *data = dz0_tcr;
    return SCPE_OK;

  case 6:
  case 7:
    *data = dz0_msr;
    return SCPE_OK;
  }
}

/*	Debug routine dz0_rd.....

t_stat dz0_rd (int32 *data, int32 PA, int32 access)
{
  t_stat foo;

  foo = dz0_rd_1 (&*data, PA, access);

  fprintf(stderr, "dz0-rd: data = %d, PA = %d, access = %d \r\n", 
		*data, PA & 7, access); 
  return foo;
}
*/

/****************************************************************/
/* Write to DZ interface.			 		*/

t_stat dz0_wr (int32 data, int32 PA, int32 access)
{
  int i;

  
 /* fprintf(stderr, "dz0-wr: data = %d, PA = %d, access = %d \r\n", 
 		data, PA & 7, access); */
 

  switch (PA & 7) {			/* decode PA<2:0> */
  case 0:
    if (access == WRITEB) {
      dz0_csr &= 0xff80;
      dz0_csr |= (data & 0x68);		/* mask out init-bit for now */
    }
    else {
      dz0_csr &= 0xbf80;
      dz0_csr |= (data & 0x5068);	/* mask out init-bit for now (5078) */
    }
    if (dz0_csr & 0xc020) {		     
      if (dz0_tcr & 0xff & enabled_tx_lines) {	
	int_req |= INT_DZ0TX;
      } else {
	int_req &= ~INT_DZ0TX;
      }
    }
    if ((dz0_csr & 0xc0) == 0xc0) {
      int_req |= INT_DZ0RX;
    } else {
      int_req &= ~INT_DZ0RX;
    }
    return SCPE_OK;

  case 1:
    dz0_csr &= 0xbfff;
    dz0_csr |= (data & 0x50) << 8;
    return SCPE_OK;

  case 2:
  case 3:				/* illegal */
    dz0_lpr = data;			/* not yet implemented */
    return SCPE_OK;

  case 4:
    i = dz0_tcr;		/* save old value */
    if (access == WRITEB) {
      dz0_tcr &= 0xff00;
      dz0_tcr |= (data & 0xff);
    }
    else {
      dz0_tcr = data;
    }

    if (modem_control) {
      dz0_msr |= ((dz0_tcr & 0xff00) & ((dz0_msr << 8) & 0xff00));
		      /* Set carrier detect bits that have RI and DTR set */
      dz0_msr &= ~(dz0_tcr >> 8);
		      /* If DTR set, clear corresponding RI bit */
      
      if (modem_control & 2) {			/* if active line disconnect */
	i = (i & ~dz0_tcr) >> 8;		/* find out what line to disconn */
	if (i) {
	  disconn_line(i);
	  dz0_msr &= ~(i << 8);
	}
      }
    }

    /* Now we shall se what line there is to enable. */

    if ((enabled_tx_lines & dz0_tcr & 0xff) == 0) {
      dz0_csr &= ~0x8000;		/* remove TRDY */
      int_req &= ~INT_DZ0TX;
      return SCPE_OK;			/* and exit if no lines enabled */
    }
    rline = 0;
    i = 1;
    while ((enabled_tx_lines & dz0_tcr & i) == 0) {
      rline++;
      i = i << 1;
    }
    dz0_csr &= 0xf8ff;
    dz0_csr |= (rline << 8) | 0x8000;
    if ((dz0_csr & 0xc020) == 0xc020) {
      if (dz0_tcr & enabled_tx_lines & 0xff) {
	int_req |= INT_DZ0TX;
	/*      fprintf(stderr, "generated dz-int"); */
      } else {
	int_req &= ~INT_DZ0TX;
      }
      /*    fprintf(stderr, " enabled line: %d csr = 0x%4x\r\n", rline, dz0_csr); */
    }
      return SCPE_OK;

  case 5:
    i = dz0_tcr;		/* save old value */
    dz0_tcr &= 0x00ff;
    dz0_tcr |= (data & 0xff) << 8;

    if (modem_control) {
      dz0_msr |= ((dz0_tcr & 0xff00) & ((dz0_msr << 8) & 0xff00));
		      /* Set carrier detect bits that have RI and DTR set */
      dz0_msr &= ~(dz0_tcr >> 8);
		      /* If DTR set, clear corresponding RI bit */

      if (modem_control & 2) {			/* if active line disconnect */
	i = (i & ~dz0_tcr) >> 8;		/* find out what line to disconn */
	if (i) {
	  disconn_line(i);
	  dz0_msr &= ~(i << 8);
	}
      }
    }
    return SCPE_OK;

  case 6:
    if (access == WRITEB) {
      dz0_tdr &= 0xff00;
      dz0_tdr |= (data & 0xff);
    }
    else {
      dz0_tdr = data;
    }
/* the simple send-routine.
    i = (dz0_csr >> 8) & 7;
    if (connection[i]) {
      write(connection[i], (char *) &data, 1);
    }
    if (dz0_tcr & 0xff) {
      int_req |= INT_DZ0TX;
    } else {
      int_req &= ~INT_DZ0TX;
    }
    return SCPE_OK;
*/
    i = (dz0_csr >> 8) & 7;		/* line to send to */
    if ((conntxbufp2[i] < MAXBUF) && connection[i]) {
      conntxbuff[i][conntxbufp2[i]] = (char) data;
      conntxbufp2[i]++;
      if (conntxbufp2[i] > MAXBUF - 10) {
	enabled_tx_lines &= ~(1 << i);	/* disable this line */
	poll_tx_channels();		/* and try to send */
      }
    }
    if (dz0_tcr & enabled_tx_lines & 0xff) {
      int_req |= INT_DZ0TX;
    } else {
      int_req &= ~INT_DZ0TX;
    }

/*    fprintf(stderr, "Sent chr: %d, line %d\r\n", data & 0xff, (dz0_csr >> 8) & 7); */
    return SCPE_OK;

  case 7: 
    dz0_tdr &= 0x00ff;
    dz0_tdr |= (data & 0xff) << 8;
    return SCPE_OK;
  }
  
  return SCPE_OK;
}

t_stat dz0_svc (UNIT *uptr)
{
  fprintf(stderr, "dz0 -svc called \n");
}

t_stat dz0_reset (DEVICE *dptr)
{
  /*  fprintf(stderr, "dz0 -reset\n"); */
  dz0_tcr &= 0xff00;
  dz0_csr = 0;
  return SCPE_OK;
}


/* This is where we initialize the network connections etc     */

t_stat dz0_attach (UNIT *uptr, char *cptr)
{
  char* next;
  int32 port, sock, i, options;
  
  port = strtoul(cptr, &next, 10);
  options = strtoul(next, NULL, 10);

  /* Linux supports 32 bit port numbers now..../TN.  */

  if (( port < 1) || (port > 65535)) {
    fprintf(stderr, "DZ0: Invalid port number.\r\n");
    return SCPE_OPENERR;
  }

  modem_control = (options & 3);	/* set modem flags */
  timer_catchup = (options & 4) << 2;	/* set time mode   */
  if (modem_control) fprintf(stderr, "Modem control activated.\n");
  if (timer_catchup) fprintf(stderr, "Timer catchup mode activated.\n");
  
  /* initialize the set of active sockets. */
  for (i = 0; i < 9; i++) connection[i] = 0;
  enabled_tx_lines = 0xff;	/* all lines enabled from the beginning. */
  disabled_telnet_bin = 0x00;	/* default bin mode on all telnet conns */
  skip_next_char = 0;		/* no skipping at startup */
  
  /* create the socket and set it up to accept connections. */
  sock = make_socket (port);
  if (sock < 0) return -1;
  if (listen (sock, 1) < 0)
    {
      perror("dz0: listen");
      exit (EXIT_FAILURE);
    }
  fprintf(stderr, "main sock is %d\r\n", sock);
  connection[8] = sock;
  
  /* setup event handlers for line clock and network I/O here */

  new_action.sa_handler = catch_alarm;
  sigemptyset(&new_action.sa_mask);
  new_action.sa_flags = SA_RESTART;

  sigaction(SIGIO, &new_action, &old_action);
  sigaction(SIGALRM, &new_action, NULL);
  
  /* setup interval timer... start with 20 milliseconds or so....   */
  /* this is the lowest naturally occuring line-freq-interrupt      */

  setup_interval_timer(dz0_norm_i_clock);
  /* old_clock = 0;		/* enable signal-driven clock */

  if (timer_catchup) {
    /* This does only work on systems having time_t == long int */
    emulated_time = time(NULL);	/* Get start value. */
  }

  /* magic mantra to make remote telnet client behave nicely */

  mantra[0] = 255;  /* IAC      */
  mantra[1] = 251;  /* WILL     */
  mantra[2] = 34;   /* LINEMODE */
  
  mantra[3] = 255;  /* IAC      */
  mantra[4] = 251;  /* WILL     */
  mantra[5] = 3;    /* SGA      */
  
  mantra[6] = 255;  /* IAC      */
  mantra[7] = 251;  /* WILL     */
  mantra[8] = 1;    /* ECHO     */

  mantra[9] = 255;   /* IAC      */
  mantra[10] = 251;  /* WILL     */
  mantra[11] = 0;    /* BINARY   */

  mantra[12] = 255;  /* IAC     */
  mantra[13] = 253;  /* DO      */
  mantra[14] = 0;    /* BINARY  */

  uptr -> flags = uptr -> flags | UNIT_ATT;
  snprintf(dz0_attach_message, 99, "port %d, options %d", port, options);
  uptr -> filename = dz0_attach_message;
  return SCPE_OK;
}

t_stat dz0_detach (UNIT *uptr)
{
  int i;

  if (uptr -> flags & UNIT_ATT) { /* done if not attached */
    for(i = 0; i < 8; i++) {
      if (connection[i]) {
#	ifdef PDP11
	 xwrite(connection[i], "\r\nPDP11 simulator shutting down... please come back later.\r\n\n\n\n");
#	endif
#	ifdef PDP10
	xwrite(connection[i], "\r\nPDP10 simulator shutting down... please come back later.\r\n\n\n\n");
#	endif
	close(connection[i]);
      }
    }
    close(connection[8]);
    /* old_clock = 1;		/* back to old clock mode */
    uptr -> flags = uptr -> flags & ~UNIT_ATT;
  }
  close (DZ0sock);
  return 0;
}
